use rltk::{GameState, Rltk, RGB, VirtualKeyCode};
use specs::prelude::*;
use std::cmp::{max, min};
use specs_derive::Component;

#[derive(Component)]
struct Position {
  x: i32,
  y: i32,
}

#[derive(Component)]
struct Renderable {
  glyph: rltk::FontCharType,
  fg: RGB,
  bg: RGB,
}

struct State {
  ecs: World,
}

impl State {
  fn run_systems(&mut self) {
    let mut mob = Mob{};
    mob.run_now(&self.ecs);
    self.ecs.maintain();
  }
}

impl GameState for State {
  fn tick(&mut self, ctx: &mut Rltk) {
    ctx.cls();
    self.run_systems();
    player_input(self, ctx);
    
    let positions = self.ecs.read_storage::<Position>();
    let renderables = self.ecs.read_storage::<Renderable>();

    for (pos, render) in (&positions, &renderables).join() {
      ctx.set(pos.x, pos.y, render.fg, render.bg, render.glyph);
    }
  }
}

fn main() -> rltk::BError {
  use rltk::RltkBuilder;
  let context = RltkBuilder::simple80x50()
    .with_title("¡Rogue-ish Rumble!")
    .build()?;
  let mut gs = State { ecs: World::new() };

  gs.ecs.register::<Position>();
  gs.ecs.register::<Renderable>();
  gs.ecs.register::<Mobile>();
  gs.ecs.register::<Player>();
  
  gs.ecs
    .create_entity()
    .with(Position { x: 40, y: 25 })
    .with(Renderable {
      glyph: rltk::to_cp437('@'),
      fg: RGB::named(rltk::GOLD),
      bg: RGB::named(rltk::BLACK),
    })
    .with(Player{})
    .build();

  for i in 0..10 {
    gs.ecs
    .create_entity()
    .with(Position { x: i * 7, y: 20 })
    .with(Renderable {
      glyph: rltk::to_cp437('^'),
      fg: RGB::named(rltk::PURPLE),
      bg: RGB::named(rltk::RED),
    })
    .with(Mobile { x: -1, y: 0 })
    .build();
  }

  rltk::main_loop(context, gs)
}

#[derive(Component)]
struct Mobile {
  x: i32,
  y: i32,
}

struct Mob {}
impl<'a> System<'a> for Mob {
  type SystemData = (ReadStorage<'a, Mobile>, WriteStorage<'a, Position>);
  fn run(&mut self, (mob, mut pos) : Self::SystemData) {
    for (mob, pos) in (&mob, &mut pos).join() {
      pos.x += mob.x;
      pos.y += mob.y;
      if pos.x < 0 { pos.x += 80; }
    }
  }
}

#[derive(Component, Debug)]
struct Player {}

fn try_move_player(dx: i32, dy: i32, ecs: &mut World) {
  let mut positions = ecs.write_storage::<Position>();
  let mut players = ecs.write_storage::<Player>();

  for(_player, pos) in (&mut players, &mut positions).join() {
    pos.x = min(79, max(0, pos.x + dx));
    pos.y = min(49, max(0, pos.y + dy));
  }
}

fn player_input(gs: &mut State, ctx: &mut Rltk) {
  match ctx.key {
    None => {},
    Some(key) => match key {
      VirtualKeyCode::Left  => try_move_player(-1,  0, &mut gs.ecs),
      VirtualKeyCode::Right => try_move_player( 1,  0, &mut gs.ecs),
      VirtualKeyCode::Up    => try_move_player( 0, -1, &mut gs.ecs),
      VirtualKeyCode::Down  => try_move_player( 0,  1, &mut gs.ecs),
      _ => {},
    },
  }
}
